/*!
 ***********************************************************************
 *  \mainpage
 *     This is the H.26L decoder reference software. For detailed documentation
 *     see the comments in each file.
 *
 *  \author
 *     The main contributors are listed in contributors.h
 *
 *  \version
 *     TML 8.4
 *
 *  \note
 *     tags are used for document system "doxygen"
 *     available at http://www.doxygen.org
 *
 *  \par Limitations:
 *     Using different NAL's the assignment of partition-id to containing
 *     syntax elements may got lost, if this information is not transmitted.
 *     The same has to be stated for the partitionlength if partitions are
 *     merged by the NAL.
 *  \par
 *     The presented solution in Q15-K-16 solves both of this problems as the
 *     departitioner parses the bitstream before decoding. Due to syntax element
 *     dependencies both, partition bounds and partitionlength information can
 *     be parsed by the departitioner.
 *
 *  \par Handling partition information in external file:
 *     As the TML is still a work in progress, it makes sense to handle this
 *     information for simplification in an external file, here called partition
 *     information file, which can be found by the extension .dp extending the
 *     original encoded H.26L bitstream. In this file partition-ids followed by its
 *     partitionlength is written. Instead of parsing the bitstream we get the
 *     partition information now out of this file.
 *     This information is assumed to be never sent over transmission channels
 *     (simulation scenarios) as it's information we allways get using a
 *     "real" departitioner before decoding
 *
 *  \par Extension of Interim File Format:
 *     Therefore a convention has to be made within the interim file format.
 *     The underlying NAL has to take care of fulfilling these conventions.
 *     All partitions have to be bytealigned to be readable by the decoder,
 *     So if the NAL-encoder merges partitions, >>this is only possible to use the
 *     VLC structure of the H.26L bitstream<<, this bitaligned structure has to be
 *     broken up by the NAL-decoder. In this case the NAL-decoder is responsable to
 *     read the partitionlength information from the partition information file.
 *     Partitionlosses are signaled with a partition of zero length containing no
 *     syntax elements.
 *
 */
/*!
 *  \file
 *     ldecod.c
 *  \brief
 *     TML decoder project main()
 *  \author
 *     Main contributors (see contributors.h for copyright, address and affiliation details)
 *     - Inge Lille-Langy       <inge.lille-langoy@telenor.com>
 *     - Rickard Sjoberg         <rickard.sjoberg@era.ericsson.se>
 *     - Stephan Wenger          <stewe@cs.tu-berlin.de>
 *     - Jani Lainema            <jani.lainema@nokia.com>
 *     - Sebastian Purreiter     <sebastian.purreiter@mch.siemens.de>
 *     - Byeong-Moon Jeon        <jeonbm@lge.com>
 *     - Gabi Blaettermann       <blaetter@hhi.de>
 *
 ***********************************************************************
 */

#include "contributors.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <sys/timeb.h>

#if defined WIN32
  #include <conio.h>
#endif

#include "global.h"
#include "elements.h"
#include "bitsbuf.h"

#define TML         "8"
#define VERSION     "8.50"
#define LOGFILE     "log.dec"
#define DATADECFILE "data.dec"
#define TRACEFILE   "trace_dec.txt"

/*!
 ***********************************************************************
 * \brief
 *    main function for TML decoder
 ***********************************************************************
 */
int main(int argc, char **argv)
{
  struct inp_par    *inp;         // input parameters from input configuration file
  struct snr_par    *snr;         // statistics
  struct img_par    *img;         // image parameters

    // allocate memory for the structures
  if ((inp =  (struct inp_par *)calloc(1, sizeof(struct inp_par)))==NULL) no_mem_exit("main: inp");
  if ((snr =  (struct snr_par *)calloc(1, sizeof(struct snr_par)))==NULL) no_mem_exit("main: snr");
  if ((img =  (struct img_par *)calloc(1, sizeof(struct img_par)))==NULL) no_mem_exit("main: img");

  // Read Configuration File
  if (argc != 2)
  {
    snprintf(errortext, ET_SIZE, "Usage: %s <config.dat> \n\t<config.dat> defines decoder parameters",argv[0]);
    error(errortext, 300);
  }

    // Initializes Configuration Parameters with configuration file
  init_conf(inp, argv[1]);

  img->UseConstrainedIntraPred = inp->UseConstrainedIntraPred;

    // Allocate Slice data struct
  malloc_slice(inp,img);

  init(img);
  img->number=0;

  // B pictures
  Bframe_ctr=0;

  // time for total decoding session
  tot_time = 0;
  while (decode_one_frame(img, inp, snr) != EOS);

    // B PICTURE : save the last P picture
  write_prev_Pframe(img, p_out);

  report(inp, img, snr);

  free_slice(inp,img);
  free_mem4global_buffers(inp, img);

  CloseBitstreamFile();

  fclose(p_out);
  fclose(p_ref);
#if TRACE
  fclose(p_trace);
#endif

  free (inp);
  free (snr);
  free (img);

  return 0;
}


/*!
 ***********************************************************************
 * \brief
 *    Initilize some arrays
 ***********************************************************************
 */
void init(struct img_par *img)  //!< image parameters
{
  int i;
  // initilize quad matrix used in snr routine

  for (i=0; i <  256; i++)
  {
    img->quad[i]=i*i; // fix from TML 1, truncation removed
  }
}

/*!
 ************************************************************************
 * \brief
 *    Read input from configuration file
 *
 * \par Input:
 *    Name of configuration filename
 *
 * \par Output
 *    none
 ************************************************************************
 */
void init_conf(struct inp_par *inp,
               char *config_filename)
{
  FILE *fd;
  int NAL_mode;

  // read the decoder configuration file
  if((fd=fopen(config_filename,"r")) == NULL)
  {
    snprintf(errortext, ET_SIZE, "Error: Control file %s not found\n",config_filename);
    error(errortext, 300);
  }

  fscanf(fd,"%s",inp->infile);                // H.26L compressed input bitsream
  fscanf(fd,"%*[^\n]");

  fscanf(fd,"%s",inp->outfile);               // YUV 4:2:2 input format
  fscanf(fd,"%*[^\n]");

  fscanf(fd,"%s",inp->reffile);               // reference file
  fscanf(fd,"%*[^\n]");

    // Symbol mode
  fscanf(fd,"%d,",&inp->symbol_mode);        // 0: UVLC 1: CABAC
  fscanf(fd,"%*[^\n]");
  if (inp->symbol_mode != UVLC && inp->symbol_mode != CABAC)
  {
    snprintf(errortext, ET_SIZE, "Unsupported symbol mode=%d, use UVLC=0 or CABAC=1",inp->symbol_mode);
    error(errortext,1);
  }

  // UseConstrainedIntraPred
  fscanf(fd,"%d,",&inp->UseConstrainedIntraPred);        // 0: UsePred   1: ConstrainPred
  fscanf(fd,"%*[^\n]");
  if(inp->UseConstrainedIntraPred != 0 && inp->UseConstrainedIntraPred != 1)
  {
    snprintf(errortext, ET_SIZE, "Unsupported value=%d on constrained intra pred",inp->UseConstrainedIntraPred);
    error(errortext,1);
  }

  // Frame buffer size
  fscanf(fd,"%d,",&inp->buf_cycle);
  fscanf(fd,"%*[^\n]");
  if (inp->buf_cycle < 1)
  {
    snprintf(errortext, ET_SIZE, "Frame Buffer Size is %d. It has to be at least 1",inp->buf_cycle);
    error(errortext,1);
  }

  fscanf(fd,"%d",&(NAL_mode));                // NAL mode
    fscanf(fd,"%*[^\n]");

    switch(NAL_mode)
  {
  case 0:
    inp->of_mode = PAR_OF_26L;
    // Note: Data Partitioning in 26L File Format not yet supported
    inp->partition_mode = PAR_DP_1;
    break;
  case 1:
    inp->of_mode = PAR_OF_SLICE;
    inp->partition_mode = PAR_DP_1;
    break;
  case 2:
    inp->of_mode = PAR_OF_SLICE;
    inp->partition_mode = PAR_DP_2;
    break;
  case 3:
    inp->of_mode = PAR_OF_SLICE;
    inp->partition_mode = PAR_DP_4;
    break;
  default:
    snprintf(errortext, ET_SIZE, "NAL mode %i is not supported", NAL_mode);
    error(errortext,400);
  }


#if TRACE
  if ((p_trace=fopen(TRACEFILE,"w"))==0)             // append new statistic at the end
  {
    snprintf(errortext, ET_SIZE, "Error open file %s!",TRACEFILE);
    error(errortext,500);
  }
#endif


  if (OpenBitstreamFile (inp->infile) < 0)
  {
    snprintf (errortext, ET_SIZE, "Cannot open bitstream file '%s'", inp->infile);
    error(errortext,500);
  }
  if ((p_out=fopen(inp->outfile,"wb"))==0)
  {
    snprintf(errortext, ET_SIZE, "Error open file %s ",inp->outfile);
    error(errortext,500);
  }

  fprintf(stdout,"--------------------------------------------------------------------------\n");
  fprintf(stdout," Decoder config file                    : %s \n",config_filename);
  fprintf(stdout,"--------------------------------------------------------------------------\n");
  fprintf(stdout," Input H.26L bitstream                  : %s \n",inp->infile);
  fprintf(stdout," Output decoded YUV 4:2:0               : %s \n",inp->outfile);
  fprintf(stdout," Output status file                     : %s \n",LOGFILE);
  if ((p_ref=fopen(inp->reffile,"rb"))==0)
  {
    fprintf(stdout," Input reference file                   : %s does not exist \n",inp->reffile);
    fprintf(stdout,"                                          SNR values are not available\n");
  }
  else
    fprintf(stdout," Input reference file                   : %s \n",inp->reffile);
  fprintf(stdout,"--------------------------------------------------------------------------\n");

  fprintf(stdout,"Frame   TR    QP  SnrY    SnrU    SnrV   Time(ms)\n");
}

/*!
 ************************************************************************
 * \brief
 *    Reports the gathered information to appropriate outputs
 *
 * \par Input:
 *    struct inp_par *inp,
 *    struct img_par *img,
 *    struct snr_par *stat
 *
 * \par Output:
 *    None
 ************************************************************************
 */
void report(struct inp_par *inp, struct img_par *img, struct snr_par *snr)
{
  #define OUTSTRING_SIZE 255
  char string[OUTSTRING_SIZE];
  FILE *p_log;

#ifndef WIN32
  time_t  now;
  struct tm *l_time;
#else
  char timebuf[128];
#endif

  fprintf(stdout,"-------------------- Average SNR all frames ------------------------------\n");
  fprintf(stdout," SNR Y(dB)           : %5.2f\n",snr->snr_ya);
  fprintf(stdout," SNR U(dB)           : %5.2f\n",snr->snr_ua);
  fprintf(stdout," SNR V(dB)           : %5.2f\n",snr->snr_va);
  fprintf(stdout," Total decoding time : %.3f sec \n",tot_time*0.001);
  fprintf(stdout,"--------------------------------------------------------------------------\n");
  fprintf(stdout," Exit TML %s decoder, ver %s \n",TML,VERSION);

  // write to log file

  snprintf(string, OUTSTRING_SIZE, "%s", LOGFILE);
  if (fopen(string,"r")==0)                    // check if file exist
  {
    if ((p_log=fopen(string,"a"))==0)
    {
      snprintf(errortext, ET_SIZE, "Error open file %s for appending",string);
      error(errortext, 500);
    }
    else                                              // Create header to new file
    {
      fprintf(p_log," ------------------------------------------------------------------------------------------\n");
      fprintf(p_log,"|  Decoder statistics. This file is made first time, later runs are appended               |\n");
      fprintf(p_log," ------------------------------------------------------------------------------------------ \n");
      fprintf(p_log,"| Date  | Time  |    Sequence        |#Img|Format|SNRY 1|SNRU 1|SNRV 1|SNRY N|SNRU N|SNRV N|\n");
      fprintf(p_log," ------------------------------------------------------------------------------------------\n");
    }
  }
  else
    p_log=fopen("log.dec","a");                    // File exist,just open for appending

#ifdef WIN32
  _strdate( timebuf );
  fprintf(p_log,"| %1.5s |",timebuf );

  _strtime( timebuf);
  fprintf(p_log," % 1.5s |",timebuf);
#else
  now = time ((time_t *) NULL); // Get the system time and put it into 'now' as 'calender time'
  time (&now);
  l_time = localtime (&now);
  strftime (string, sizeof string, "%d-%b-%Y", l_time);
  fprintf(p_log,"| %1.5s |",string );

  strftime (string, sizeof string, "%H:%M:%S", l_time);
  fprintf(p_log,"| %1.5s |",string );
#endif

  fprintf(p_log,"%20.20s|",inp->infile);

  fprintf(p_log,"%3d |",img->number);

  fprintf(p_log,"%6.3f|",snr->snr_y1);
  fprintf(p_log,"%6.3f|",snr->snr_u1);
  fprintf(p_log,"%6.3f|",snr->snr_v1);
  fprintf(p_log,"%6.3f|",snr->snr_ya);
  fprintf(p_log,"%6.3f|",snr->snr_ua);
  fprintf(p_log,"%6.3f|\n",snr->snr_va);

  fclose(p_log);

  snprintf(string, OUTSTRING_SIZE,"%s", DATADECFILE);
  p_log=fopen(string,"a");

  if(Bframe_ctr != 0) // B picture used
  {
    fprintf(p_log, "%3d %2d %2d %2.2f %2.2f %2.2f %5d "
      "%2.2f %2.2f %2.2f %5d "
      "%2.2f %2.2f %2.2f %5d %.3f\n",
      img->number, 0, img->qp,
      snr->snr_y1,
      snr->snr_u1,
      snr->snr_v1,
      0,
      0.0,
      0.0,
      0.0,
      0,
      snr->snr_ya,
      snr->snr_ua,
      snr->snr_va,
      0,
      (double)0.001*tot_time/(img->number+Bframe_ctr-1));
  }
  else
  {
    fprintf(p_log, "%3d %2d %2d %2.2f %2.2f %2.2f %5d "
      "%2.2f %2.2f %2.2f %5d "
      "%2.2f %2.2f %2.2f %5d %.3f\n",
      img->number, 0, img->qp,
      snr->snr_y1,
      snr->snr_u1,
      snr->snr_v1,
      0,
      0.0,
      0.0,
      0.0,
      0,
      snr->snr_ya,
      snr->snr_ua,
      snr->snr_va,
      0,
      (double)0.001*tot_time/img->number);
  }
  fclose(p_log);
}

/*!
 ************************************************************************
 * \brief
 *    Allocates the slice structure along with its dependent
 *    data structures
 *
 * \par Input:
 *    Input Parameters struct inp_par *inp,  struct img_par *img
 ************************************************************************
 */
void malloc_slice(struct inp_par *inp, struct img_par *img)
{
  int i;
  DataPartition *dataPart;
  Slice *currSlice;
  const int buffer_size = MAX_CODED_FRAME_SIZE; // picture size unknown at this time, this value is to check

  switch(inp->of_mode) // init depending on NAL mode
  {
    case PAR_OF_26L:
      // Current File Format
      img->currentSlice = (Slice *) calloc(1, sizeof(Slice));
      if ( (currSlice = img->currentSlice) == NULL)
      {
        snprintf(errortext, ET_SIZE, "Memory allocation for Slice datastruct in NAL-mode %d failed", inp->of_mode);
        error(errortext,100);
      }
      if (inp->symbol_mode == CABAC)
      {
        // create all context models
        currSlice->mot_ctx = create_contexts_MotionInfo();
        currSlice->tex_ctx = create_contexts_TextureInfo();
      }

      switch(inp->partition_mode)
      {
      case PAR_DP_1:
        currSlice->max_part_nr = 1;
        break;
      case PAR_DP_2:
        error("Data Partitioning Mode 2 in 26L-Format not yet supported",1);
        break;
      case PAR_DP_4:
        error("Data Partitioning Mode 4 in 26L-Format not yet supported",1);
        break;
      default:
        error("Data Partitioning Mode not supported!",1);
        break;
      }

      currSlice->partArr = (DataPartition *) calloc(1, sizeof(DataPartition));
      if (currSlice->partArr == NULL)
      {
        snprintf(errortext, ET_SIZE, "Memory allocation for Data Partition datastruct in NAL-mode %d failed", inp->of_mode);
        error(errortext, 100);
      }
      dataPart = currSlice->partArr;
      dataPart->bitstream = (Bitstream *) calloc(1, sizeof(Bitstream));
      if (dataPart->bitstream == NULL)
      {
        snprintf(errortext, ET_SIZE, "Memory allocation for Bitstream datastruct in NAL-mode %d failed", inp->of_mode);
        error(errortext, 100);
      }
      dataPart->bitstream->streamBuffer = (byte *) calloc(buffer_size, sizeof(byte));
      if (dataPart->bitstream->streamBuffer == NULL)
      {
        snprintf(errortext, ET_SIZE, "Memory allocation for bitstream buffer in NAL-mode %d failed", inp->of_mode);
        error(errortext, 100);
      }
      return;
    case PAR_OF_SLICE:
      // NAL File Format
      img->currentSlice = (Slice *) calloc(1, sizeof(Slice));
      if ( (currSlice = img->currentSlice) == NULL)
      {
        snprintf(errortext, ET_SIZE, "Memory allocation for Slice datastruct in NAL-mode %d failed", inp->of_mode);
        error(errortext, 100);
      }

      if (inp->symbol_mode == CABAC)
      {
        // create all context models
        currSlice->mot_ctx = create_contexts_MotionInfo();
        currSlice->tex_ctx = create_contexts_TextureInfo();
      }

      switch(inp->partition_mode)
      {
      case PAR_DP_1:
        currSlice->max_part_nr = 1;
        break;
      case PAR_DP_2:
        currSlice->max_part_nr = 2;
        break;
      case PAR_DP_4:
        currSlice->max_part_nr = 4;
        break;
      default:
        error("Data Partitioning Mode not supported!",1);
        break;
      }

      currSlice->partArr = (DataPartition *) calloc(currSlice->max_part_nr, sizeof(DataPartition));
      if (currSlice->partArr == NULL)
      {
        snprintf(errortext, ET_SIZE, "Memory allocation for Data Partition datastruct in NAL-mode %d failed", inp->of_mode);
        error(errortext, 100);
      }

      for (i=0; i<currSlice->max_part_nr; i++) // loop over all data partitions
      {
        dataPart = &(currSlice->partArr[i]);
        dataPart->bitstream = (Bitstream *) calloc(1, sizeof(Bitstream));
        if (dataPart->bitstream == NULL)
        {
          snprintf(errortext, ET_SIZE, "Memory allocation for Bitstream datastruct in NAL-mode %d failed", inp->of_mode);
          error(errortext, 100);
        }
        dataPart->bitstream->streamBuffer = (byte *) calloc(buffer_size, sizeof(byte));
        if (dataPart->bitstream->streamBuffer == NULL)
        {
          snprintf(errortext, ET_SIZE, "Memory allocation for bitstream buffer in NAL-mode %d failed", inp->of_mode);
          error(errortext, 100);
        }
      }
      return;
    default:
      snprintf(errortext, ET_SIZE, "Output File Mode %d not supported", inp->of_mode);
      error(errortext, 600);
  }

}

/*!
 ************************************************************************
 * \brief
 *    Memory frees of the Slice structure and of its dependent
 *    data structures
 *
 * \par Input:
 *    Input Parameters struct inp_par *inp,  struct img_par *img
 ************************************************************************
 */
void free_slice(struct inp_par *inp, struct img_par *img)
{
  int i;
  DataPartition *dataPart;
  Slice *currSlice = img->currentSlice;

  switch(inp->of_mode) // init depending on NAL mode
  {
    case PAR_OF_26L:
      // Current File Format
      dataPart = currSlice->partArr;  // only one active data partition
      if (dataPart->bitstream->streamBuffer != NULL)
        free(dataPart->bitstream->streamBuffer);
      if (dataPart->bitstream != NULL)
        free(dataPart->bitstream);
      if (currSlice->partArr != NULL)
        free(currSlice->partArr);
      if (inp->symbol_mode == CABAC)
      {
        // delete all context models
        delete_contexts_MotionInfo(currSlice->mot_ctx);
        delete_contexts_TextureInfo(currSlice->tex_ctx);
      }
      if (currSlice != NULL)
        free(img->currentSlice);
      break;
    case PAR_OF_SLICE:
      // NAL File Format
      for (i=0; i<currSlice->max_part_nr; i++) // loop over all data partitions
      {
        dataPart = &(currSlice->partArr[i]);
        if (dataPart->bitstream->streamBuffer != NULL)
          free(dataPart->bitstream->streamBuffer);
        if (dataPart->bitstream != NULL)
          free(dataPart->bitstream);
      }
      if (currSlice->partArr != NULL)
        free(currSlice->partArr);
      if (inp->symbol_mode == CABAC)
      {
        // delete all context models
        delete_contexts_MotionInfo(currSlice->mot_ctx);
        delete_contexts_TextureInfo(currSlice->tex_ctx);
      }
      if (currSlice != NULL)
        free(img->currentSlice);
      break;
    default:
      snprintf(errortext, ET_SIZE,  "Output File Mode %d not supported", inp->of_mode);
      error(errortext, 400);
  }

}

/*!
 ************************************************************************
 * \brief
 *    Dynamic memory allocation of frame size related global buffers
 *    buffers are defined in global.h, allocated memory must be freed in
 *    void free_mem4global_buffers()
 *
 *  \par Input:
 *    Input Parameters struct inp_par *inp, Image Parameters struct img_par *img
 *
 *  \par Output:
 *     Number of allocated bytes
 ***********************************************************************
 */
int get_mem4global_buffers(struct inp_par *inp, struct img_par *img)
{
  int j,memory_size=0;
#ifdef _ADAPT_LAST_GROUP_
  extern int *last_P_no;
#endif

  img->buf_cycle = inp->buf_cycle+1;

#ifdef _ADAPT_LAST_GROUP_
  if ((last_P_no = (int*)malloc(img->buf_cycle*sizeof(int))) == NULL)
    no_mem_exit("get_mem4global_buffers: last_P_no");
#endif

  // allocate memory for encoding frame buffers: imgY, imgUV
  // byte imgY[288][352];
  // byte imgUV[2][144][176];
  memory_size += get_mem2D(&imgY, img->height, img->width);
  memory_size += get_mem3D(&imgUV, 2, img->height_cr, img->width_cr);

  // allocate memory for multiple ref. frame buffers: mref, mcref
  // rows and cols for croma component mcef[ref][croma][4x][4y] are switched
  // compared to luma mref[ref][4y][4x] for whatever reason
  // number of reference frames increased by one for next P-frame
  memory_size += get_mem3D(&mref, img->buf_cycle+1, img->height, img->width);

  if((mcef = (byte****)calloc(img->buf_cycle+1,sizeof(byte***))) == NULL)
    no_mem_exit("get_mem4global_buffers: mcef");
  for(j=0;j<img->buf_cycle+1;j++)
        memory_size += get_mem3D(&(mcef[j]), 2, img->height_cr*2, img->width_cr*2);

  // allocate memory for imgY_prev
  // byte imgY_prev[288][352];
  // byte imgUV_prev[2][144][176];
  memory_size += get_mem2D(&imgY_prev, img->height, img->width);
  memory_size += get_mem3D(&imgUV_prev, 2, img->height_cr, img->width_cr);

  // allocate memory for reference frames of each block: refFrArr
  // int  refFrArr[72][88];
  memory_size += get_mem2Dint(&refFrArr, img->height/BLOCK_SIZE, img->width/BLOCK_SIZE);

  // allocate memory for filter strength for 4x4 lums and croma blocks:loopb, loopc
  //byte loopb[92][76]; i.e.[x][y]
  //byte loopc[48][40];
  #if !defined LOOP_FILTER_MB
    memory_size += get_mem2D(&loopb, img->width/BLOCK_SIZE+4, img->height/BLOCK_SIZE+4);
    memory_size += get_mem2D(&loopc, img->width_cr/BLOCK_SIZE+4, img->height_cr/BLOCK_SIZE+4);
  #endif
  // allocate memory for reference frame in find_snr
    // byte imgY_ref[288][352];
  // byte imgUV_ref[2][144][176];
  memory_size += get_mem2D(&imgY_ref, img->height, img->width);
  memory_size += get_mem3D(&imgUV_ref, 2, img->height_cr, img->width_cr);

  // allocate memory for loop_filter
  // byte imgY_tmp[288][352];                      /* temp luma image
  // byte imgUV_tmp[2][144][176];                  /* temp chroma image
  memory_size += get_mem2D(&imgY_tmp, img->height, img->width);
  memory_size += get_mem3D(&imgUV_tmp, 2, img->height_cr, img->width_cr);

  // allocate memory in structure img
  if(((img->mb_data) = (Macroblock *) calloc((img->width/MB_BLOCK_SIZE) * (img->height/MB_BLOCK_SIZE),sizeof(Macroblock))) == NULL)
    no_mem_exit("get_mem4global_buffers: img->mb_data");
  if(((img->slice_numbers) = (int *) calloc((img->width/MB_BLOCK_SIZE) * (img->height/MB_BLOCK_SIZE),sizeof(int))) == NULL)
    no_mem_exit("get_mem4global_buffers: img->slice_numbers");
  if(img->UseConstrainedIntraPred)
  {
    if(((img->intra_mb) = (int *) calloc(img->width/MB_BLOCK_SIZE * img->height/MB_BLOCK_SIZE,sizeof(int))) == NULL)
      no_mem_exit("get_mem4global_buffers: img->intra_mb");
  }
  // img => int mv[92][72][3]
  memory_size += get_mem3Dint(&(img->mv),img->width/BLOCK_SIZE +4, img->height/BLOCK_SIZE,3);
  // img => int ipredmode[90][74]
  memory_size += get_mem2Dint(&(img->ipredmode),img->width/BLOCK_SIZE +2 , img->height/BLOCK_SIZE +2);
  // int dfMV[92][72][3];
  memory_size += get_mem3Dint(&(img->dfMV),img->width/BLOCK_SIZE +4, img->height/BLOCK_SIZE,3);
  // int dbMV[92][72][3];
  memory_size += get_mem3Dint(&(img->dbMV),img->width/BLOCK_SIZE +4, img->height/BLOCK_SIZE,3);
  // int fw_refFrArr[72][88];
  memory_size += get_mem2Dint(&(img->fw_refFrArr),img->height/BLOCK_SIZE,img->width/BLOCK_SIZE);
  // int bw_refFrArr[72][88];
  memory_size += get_mem2Dint(&(img->bw_refFrArr),img->height/BLOCK_SIZE,img->width/BLOCK_SIZE);
  // int fw_mv[92][72][3];
  memory_size += get_mem3Dint(&(img->fw_mv),img->width/BLOCK_SIZE +4, img->height/BLOCK_SIZE,3);
  // int bw_mv[92][72][3];
  memory_size += get_mem3Dint(&(img->bw_mv),img->width/BLOCK_SIZE +4, img->height/BLOCK_SIZE,3);
  return (memory_size);
}

/*!
 ************************************************************************
 * \brief
 *    Free allocated memory of frame size related global buffers
 *    buffers are defined in global.h, allocated memory is allocated in
 *    int get_mem4global_buffers()
 *
 * \par Input:
 *    Input Parameters struct inp_par *inp, Image Parameters struct img_par *img
 *
 * \par Output:
 *    none
 *
 ************************************************************************
 */
void free_mem4global_buffers(struct inp_par *inp, struct img_par *img)
{
  int  j;
#ifdef _ADAPT_LAST_GROUP_
  extern int *last_P_no;
  free (last_P_no);
#endif

  if(imgY[0]      != NULL) free(imgY[0]);
  if(imgY         != NULL) free(imgY);
  if(imgUV[0][0]  != NULL) free(imgUV[0][0]);
  if(imgUV[1][0]  != NULL) free(imgUV[1][0]);
  if(imgUV[0]     != NULL) free(imgUV[0]);
  if(imgUV[1]     != NULL) free(imgUV[1]);
  if(imgUV        != NULL) free(imgUV);
  if(imgY_prev[0] != NULL) free(imgY_prev[0]);
  if(imgY_prev    != NULL) free(imgY_prev);
  if(imgUV_prev[0][0] != NULL) free(imgUV_prev[0][0]);
  if(imgUV_prev[1][0] != NULL) free(imgUV_prev[1][0]);
  if(imgUV_prev[0]    != NULL) free(imgUV_prev[0]);
  if(imgUV_prev[1]    != NULL) free(imgUV_prev[1]);
  if(imgUV_prev       != NULL) free(imgUV_prev);

  // free multiple ref frame buffers:  number increased by one for next P-frame
  for(j=0;j<=img->buf_cycle;j++)
  {
    if(mref[j][0] != NULL) free(mref[j][0]);
    if(mref[j] != NULL) free(mref[j]);

    if(mcef[j][0][0] != NULL) free(mcef[j][0][0]);
    if(mcef[j][1][0] != NULL) free(mcef[j][1][0]);
    if(mcef[j][0]    != NULL) free(mcef[j][0]);
    if(mcef[j][1]    != NULL) free(mcef[j][1]);
    if(mcef[j]       != NULL) free(mcef[j]);
  };

  if(mref != NULL) free(mref);
  if(mcef != NULL) free(mcef);

  if(refFrArr[0] != NULL) free(refFrArr[0]);
  if(refFrArr    != NULL) free(refFrArr);

  #if !defined LOOP_FILTER_MB
    if(loopb[0]    != NULL) free(loopb[0]); // free loop filter strength buffer for 4x4 blocks
    if(loopb       != NULL) free(loopb);
    if(loopc[0]    != NULL) free(loopc[0]);
    if(loopc       != NULL) free(loopc);
  #endif

  if(imgY_ref[0] != NULL) free(imgY_ref[0]);    // find_snr
  if(imgY_ref    != NULL) free(imgY_ref);
  if(imgUV_ref[0][0] != NULL) free(imgUV_ref[0][0]);
  if(imgUV_ref[1][0] != NULL) free(imgUV_ref[1][0]);
  if(imgUV_ref[0]    != NULL) free(imgUV_ref[0]);
  if(imgUV_ref[1]    != NULL) free(imgUV_ref[1]);
  if(imgUV_ref       != NULL) free(imgUV_ref);
  if(imgY_tmp[0]     != NULL) free(imgY_tmp[0]);    // loop_filter
  if(imgY_tmp        != NULL) free(imgY_tmp);
  if(imgUV_tmp[0][0] != NULL) free(imgUV_tmp[0][0]);
  if(imgUV_tmp[1][0] != NULL) free(imgUV_tmp[1][0]);
  if(imgUV_tmp[0]    != NULL) free(imgUV_tmp[0]);
  if(imgUV_tmp[1]    != NULL) free(imgUV_tmp[1]);
  if(imgUV_tmp       != NULL) free(imgUV_tmp);

  // free mem, allocated for structure img
  if (img->mb_data       != NULL) free(img->mb_data);
  if (img->slice_numbers != NULL) free(img->slice_numbers);
  if(img->UseConstrainedIntraPred)
    if (img->intra_mb    != NULL) free(img->intra_mb);

  for(j=0 ; j<(img->width/BLOCK_SIZE + 4) ; j++)         // img => int mv[92][72][3]
  {
    if(img->mv[j][0] != NULL) free(img->mv[j][0]) ;
    if(img->mv[j] != NULL) free(img->mv[j]);
  }
  if(img->mv    != NULL) free(img->mv);

  if(img->ipredmode[0] != NULL) free(img->ipredmode[0]); // img => int ipredmode[90][74]
  if(img->ipredmode != NULL) free(img->ipredmode);

  for(j=0;j<(img->width/BLOCK_SIZE + 4);j++)             // int dfMV[92][72][3];
  {
    if(img->dfMV[j][0] != NULL) free(img->dfMV[j][0]);
    if(img->dfMV[j] != NULL) free(img->dfMV[j]);
  }
  if(img->dfMV    != NULL) free(img->dfMV);

  for(j=0;j<(img->width/BLOCK_SIZE + 4);j++)             // int dbMV[92][72][3];
  {
    if(img->dbMV[j][0] != NULL) free(img->dbMV[j][0]);
    if(img->dbMV[j] != NULL) free(img->dbMV[j]);
  }
  if(img->dbMV    != NULL) free(img->dbMV);

  if(img->fw_refFrArr[0] != NULL) free(img->fw_refFrArr[0]);    // int fw_refFrArr[72][88];
  if(img->fw_refFrArr    != NULL) free(img->fw_refFrArr);
  if(img->bw_refFrArr[0] != NULL) free(img->bw_refFrArr[0]);    // int bw_refFrArr[72][88];
  if(img->bw_refFrArr    != NULL) free(img->bw_refFrArr);

  for(j=0;j<(img->width/BLOCK_SIZE + 4);j++)                    // int fw_mv[92][72][3];
  {
   if(img->fw_mv[j][0] != NULL) free(img->fw_mv[j][0]);
   if(img->fw_mv[j]    != NULL) free(img->fw_mv[j]);
  }
  if(img->fw_mv        != NULL) free(img->fw_mv);

  for(j=0;j<(img->width/BLOCK_SIZE + 4);j++)                     // int bw_mv[92][72][3];
  {
    if(img->bw_mv[j][0] != NULL) free(img->bw_mv[j][0]);
    if(img->bw_mv[j] != NULL) free(img->bw_mv[j]);
  }
  if(img->bw_mv != NULL) free(img->bw_mv);

}

/*!
 ************************************************************************
 * \brief
 *    Allocate 2D memory array -> unsigned char array2D[rows][columns]
 *
 * \par Output:
 *    memory size in bytes
 ************************************************************************/
// Change 9-Aug-2001 P. List: dont allocate independant row arrays anymore
// but one complete array and move row-pointers to array. Now you can step
// to the next line with an offset of img->width
int get_mem2D(byte ***array2D, int rows, int columns)
{
  int i;

  if((*array2D      = (byte**)calloc(rows,        sizeof(byte*))) == NULL)
    no_mem_exit("get_mem2D: array2D");
  if(((*array2D)[0] = (byte* )calloc(columns*rows,sizeof(byte ))) == NULL)
    no_mem_exit("get_mem2D: array2D");

  for(i=1;i<rows;i++)
    (*array2D)[i] = (*array2D)[i-1] + columns ;

  return rows*columns;
}

/*!
 ************************************************************************
 * \brief
 *    Allocate 2D memory array -> int array2D[rows][columns]
 *
 * \par Output:
 *    memory size in bytes
 ************************************************************************
 */
// same change as in get_mem2Dint
int get_mem2Dint(int ***array2D, int rows, int columns)
{
  int i;

  if((*array2D      = (int**)calloc(rows,        sizeof(int*))) == NULL)
    no_mem_exit("get_mem2Dint: array2D");
  if(((*array2D)[0] = (int* )calloc(rows*columns,sizeof(int ))) == NULL)
    no_mem_exit("get_mem2Dint: array2D");

  for(i=1 ; i<rows ; i++)
    (*array2D)[i] =  (*array2D)[i-1] + columns  ;

  return rows*columns*sizeof(int);
}

/*!
 ************************************************************************
 * \brief
 *    Allocate 3D memory array -> unsigned char array3D[frames][rows][columns]
 *
 * \par Output:
 *    memory size in bytes
 ************************************************************************
 */
// same change as in get_mem2Dint
int get_mem3D(byte ****array3D, int frames, int rows, int columns)
{
  int  j;

  if(((*array3D) = (byte***)calloc(frames,sizeof(byte**))) == NULL)
    no_mem_exit("get_mem3D: array3D");

  for(j=0;j<frames;j++)
    get_mem2D( (*array3D)+j, rows, columns ) ;

  return frames*rows*columns;
}

/*!
 ************************************************************************
 * \brief
 *    Allocate 3D memory array -> int array3D[frames][rows][columns]
 *
 * \par Output:
 *    memory size in bytes
 ************************************************************************
 */
// same change as in get_mem2Dint
int get_mem3Dint(int ****array3D, int frames, int rows, int columns)
{
  int  j;

  if(((*array3D) = (int***)calloc(frames,sizeof(int**))) == NULL)
    no_mem_exit("get_mem3Dint: array3D");

  for(j=0;j<frames;j++)
    get_mem2Dint( (*array3D)+j, rows, columns ) ;

  return frames*rows*columns*sizeof(int);
}

/*!
 ************************************************************************
 * \brief
 *    Exit program if memory allocation failed (using error())
 * \param where
 *    string indicating which memory allocation failed
 ************************************************************************
 */
void no_mem_exit(char *where)
{
   snprintf(errortext, ET_SIZE, "Could not allocate memory: %s",where);
   error (errortext, 100);
}
